/*
 * SOURCE:  smakec.c
 * PROJECT: EasyTeX
 *
 * PURPOSE:
 *
 * UPDATES: 03/21/1994 - extracted from texerr.c
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

extern char *TmpPath;   // defined in main.c
extern struct Editor ed;

static BOOL _issection(char *line);
static BOOL _check_len(char *line, int add);

static unsigned int _read_map_line( int fptr, char _far *buf, unsigned int maxlen );
static BOOL _read_map_file(int fptr);
static long _seek_map_file(int fptr, unsigned long offset, unsigned int flags);
static void _close_map_file(int fptr);

/*
 * The following routine passes a formatted string to dst.
 * Format arguments:
 *          %% --- the '%' char
 *          %f --- the TeX format file (obsolete)
 *          %t --- the temporary directory
 *          %b --- the current file's basename
 *          %p --- the current file's path
 *          %e --- the current file's extension
 *          %/ --- the current file's path name in TeX manner
 *          %(name) --- the virtual file's name (if avaible)
 *                      The virtual file will be created according to 'name'.
 */
BOOL smakec( char *dst, char *fmt, char *fname) {

    static BOOL _get_section(char *dst, int fptr, int *ftmp, char *section);

    char base[_MAX_FNAME], ext[_MAX_EXT], path[_MAX_DIR], texp[_MAX_DIR];
    char *cp;
    char section[EDIT_WIDTH];
    unsigned long flags;
    char map[_MAX_PATH];
    extern char *MyPath;
    int fptr, ftmp = -1;

    static char *empty = "";

    *dst = '\0';

    if( fname == NULL )
        fname = GetProfileString( PR_FILENAME( ed.CurrFile ), &profnull );

    splitpath( fname, path, base, ext );
    if( *path && path[strlen(path)-1] != '\\' )
        strcat( path, "\\" );

    strcpy( texp, path );
    for( cp = texp; *cp; ++cp ) {
        if( *cp == '\\' ) *cp = '/';
    }

    strcpy(map, MyPath);
    strcat(map, "et.map");
    fptr = Open(map, ACCESS_READ);
    if (fptr==-1) {
        DoMessageBox( DMB_OK, HLP_CMDLINEERR,
                  "Warning", "Could not open map file\n'%s'", map );
        return FALSE;
    }
    /* Try to read map file to buffer */
    if(_read_map_file(fptr)) {
        Close(fptr);
        fptr=-1;
    }

    if (!_get_section(dst, fptr, &ftmp, "all")) {
        _close_map_file(fptr);
        if (ftmp!=-1) {
            Close(ftmp);
            Delete(GetProfileString( PR_RESPONSEFILE, NULL ));
        }
        return FALSE;
    }

    while( *fmt ) {
        if( *fmt == '%' ) {
            ++fmt;
            switch( *fmt ) {
                case '%': strcat(dst, "%");
                          break;

                case 'f': flags = GetProfileLong( PR_TEXFLAGS );
                          sprintf( dst+strlen(dst), "%splain%s",
                                    flags & TEX_PLAIN ? empty : "l",
                                    flags & TEX_GERMAN ? "g" : empty );
                          break;
                case 't': strcat( dst, TmpPath );
                          if( dst[strlen( dst )-1] != '\\' )
                              strcat( dst, "\\" );
                          break;
                case 'b': strcat( dst, base );
                          break;
                case 'p': strcat( dst, path );
                          break;
                case 'e': strcat( dst, ext );
                          break;
                case '/': strcat( dst, texp );
                          break;

                case '(': /* Get section name. */
                          ++fmt;
                          for(cp=section; *fmt!=')';) {
                            if (*fmt==0 || *fmt==' ') {
                                /* missing ')' */
                                DoErrorBox( HLP_CMDLINEERR,
                                    "Command Line Error:\n"
                                    "Missing ')'.");
                                return FALSE;
                            }
                            *cp++ = *fmt++;
                          }
                          *cp = 0;
                          if (!_get_section(dst, fptr, &ftmp, section)) {
                              _close_map_file(fptr);
                              if (ftmp!=-1) {
                                  Close(ftmp);
                                  Delete(GetProfileString( PR_RESPONSEFILE, NULL ));
                              }
                              return FALSE;
                          }
                          break;

                default:  break;
            }
        } else {
            strncat( dst, fmt, 1 );
        }
        ++fmt;
    }

    _close_map_file(fptr);
    if (ftmp!=-1) Close(ftmp);
    return TRUE;
}

static BOOL _get_section(char *dst, int fptr, int *ftmp, char *section) {

    static BOOL _read_section(char *dst, int fptr, int *ftmp);

    char buffer[EDIT_WIDTH];

    while(*section==' ') ++section;

    _seek_map_file(fptr, 0, SEEK_START);

    /* Search for the desired section */
    while(_read_map_line(fptr, buffer, EDIT_WIDTH)) {
        if (_issection(buffer)) {
            if (strncmp(buffer+1, section, strlen(section))==0)
                return _read_section(dst, fptr, ftmp);
        }
    }
    DoErrorBox(HLP_CMDLINEERR, "This section is unknown:\n"
                               "'%s'", section);
    return FALSE;
}

static BOOL _read_section(char *dst, int fptr, int *ftmp) {

    static BOOL _rf_use(char *dst, char *line, int fptr, int *ftmp);
    static BOOL _rf_write(char *dst, char *line, int *ftmp);
    static BOOL _rf_arg(char *dst, char *line);
    static BOOL _rf_set(char *line);

    char buffer[EDIT_WIDTH];
    BOOL ret;
    long pos;

    while(_read_map_line(fptr, buffer, EDIT_WIDTH)) {
        if (buffer[0]=='#') continue;
        if (_issection(buffer))
            break;
        if (strncmp(buffer, "write ", 6)==0)
            ret = _rf_write(dst, buffer+6, ftmp);
        else if (strncmp(buffer, "arg ", 4)==0)
            ret = _rf_arg(dst, buffer+4);
        else if (strncmp(buffer, "set ", 4)==0)
            ret = _rf_set(buffer+4);
        else if (strncmp(buffer, "use ", 4)==0) {
            pos = _seek_map_file(fptr, 0L, SEEK_CURRENT);
            ret = _rf_use(dst, buffer+4, fptr, ftmp);
            _seek_map_file(fptr, pos, SEEK_START);
        }
        if (ret==FALSE) return FALSE;
    }
    return TRUE;

}

static BOOL _rf_set(char *line) {

    static BOOL _get_args(char *dst, char **line, char delimiter);

    char buffer[EDIT_WIDTH];
    char *cp;

    if (!_get_args(buffer, &line, 0)) return FALSE;

    if (strlen(buffer)>=BUFSIZE) {
        buffer[73]=0;
        DoMessageBox( DMB_OK, HLP_CMDLINEERR,
            "Warning", "Environment string exceeds %d chars:\n"
                       "'%s'", BUFSIZE-1, buffer );
        return FALSE;
    }

    for( cp=buffer; *cp && *cp!='='; ++cp )
        *cp = toupper(*cp);

    if (!(cp = strdup(buffer))) {
        DoMessageBox( DMB_OK, HLP_CMDLINEERR,
            "Warning", "Not enough memory to store environment string:\n'%s'", buffer );
        return FALSE;
    }
    if (_putenv(cp)==0) return TRUE;
    DoMessageBox( DMB_OK, HLP_CMDLINEERR,
       "Warning", "Error in environment string:\n'%s'", buffer );

    return FALSE;
}

static BOOL _rf_use(char *dst, char *line, int fptr, int *ftmp) {

    static BOOL _get_args(char *dst, char **line, char delimiter);

    char buffer[EDIT_WIDTH];

    if (!_get_args(buffer, &line, 0)) return FALSE;
    if (*buffer) return _get_section(dst, fptr, ftmp, buffer);
}

/* Write contents to response file. */
static BOOL _rf_write(char *dst, char *line, int *ftmp) {

    static BOOL _get_args(char *dst, char **line, char delimiter);

    char buffer[EDIT_WIDTH];
    char tmpfile[_MAX_PATH];

    if (!_get_args(buffer, &line, 0)) return FALSE;
    if (*buffer == 0) return TRUE;

    if (*ftmp == -1) {
        *ftmp = CreateTmp( tmpfile, 0);
        if (*ftmp==-1) {
            DoMessageBox( DMB_OK, HLP_CMDLINEERR,
                  "Warning", "Could not open temporary response file\n'%s'", tmpfile );
            return FALSE;
        }
        PutProfileString( PR_RESPONSEFILE, tmpfile );
    }

    _Write(*ftmp, buffer, strlen(buffer));
    _Write(*ftmp, "\x0d\x0a", 2);

    return TRUE;
}

/* Add arguments to response file. */
static BOOL _rf_arg(char *dst, char *line) {

    static BOOL _get_args(char *dst, char **line, char delimiter);

    char buffer[EDIT_WIDTH];

    if (!_get_args(buffer, &line, 0)) return FALSE;
    strcat(dst, buffer);

    return TRUE;
}


/* read and expand the arguments of a command */
static BOOL _get_args(char *dst, char **line, char delimiter) {

    static BOOL _get_literal(char *buf, char **src);
    static BOOL _get_flag(char *dst, char **src );
    static BOOL _get_resword(char *dst, char **src );

    char buffer[EDIT_WIDTH];
    char literal[EDIT_WIDTH];
    BOOL answer;
    BOOL negate = FALSE;

    for(*dst = 0;**line && (**line!=delimiter);) {
        if ((**line==' ') || (**line=='\t') ) {
            ++*line;
            continue;
        }
        if (**line=='!') {
            ++*line;
            negate = TRUE;
            continue;
        }
        if (_get_literal(literal, line)) {
            /* Literal string. */
            if (!_check_len(dst, strlen(literal))) return FALSE;
            strcat(dst, literal);
        } else if (**line=='=') {
            /* Comparison. */
            ++*line;
            SKIPBLK(*line);
            _get_resword(buffer, line);
            SKIPBLK(*line);
            if (!_get_literal(literal, line)) break;
            answer = strcmp(buffer, literal);
            SKIPBLK(*line);
            if (!_get_args(literal, line, ':')) return FALSE;
            ++*line;
            if ((answer==0)^negate) {
                if (!_check_len(dst,strlen(literal))) return FALSE;
                strcat(dst, literal);
            }
            SKIPBLK(*line);
            if (!_get_args(literal, line, 0)) return FALSE;
            if ((answer!=0)^negate) {
                if (!_check_len(dst,strlen(literal))) return FALSE;
                strcat(dst, literal);
            }
            negate = FALSE;
        } else if (**line=='?') {
            /* Flag. */
            ++*line;
            SKIPBLK(*line);
            answer = _get_flag(buffer, line);
            SKIPBLK(*line);
            if(!_get_args(literal, line, ':')) return FALSE;
            if (**line) ++*line;
            if ((answer==TRUE)^negate) {
                if (!_check_len(dst,strlen(literal))) return FALSE;
                strcat(dst, literal);
            }
            SKIPBLK(*line);
            if(!_get_args(literal, line, 0)) return FALSE;
            if ((answer==FALSE)^negate) {
                if (!_check_len(dst,strlen(literal))) return FALSE;
                strcat(dst, literal);
            }
            negate = FALSE;
        } else {
            /* Reserved word. */
            _get_resword(buffer, line);
            if (!_check_len(dst,strlen(buffer))) return FALSE;
            strcat(dst, buffer);
        }
    }
    return TRUE;
}

static BOOL _get_literal(char *buf, char **src) {

    if (**src != '"') return FALSE;
    for(++*src; **src!='"'; ++*src) {
        if (**src==0) return FALSE;
        *buf++ = **src;
    }
    ++*src;
    *buf = 0;
    return TRUE;
}

static BOOL _get_flag(char *dst, char **src ) {

    static struct {
        char *Name;
        unsigned int Offset;
        unsigned int BitNum;
    } FlagArray[] = {
        { "PagesOdd", PR_P_FLAGS, 0 },
        { "PagesEven", PR_P_FLAGS, 1 },
        { "PagesAll", PR_P_FLAGS, 2 },
        { "PagesBackwards", PR_P_FLAGS, 3 },
        { "UsePagesWithPreview", PR_P_FLAGS, 4 },

        { "PrintXonXoff", PR_PRINTFLAGS, 0 },
        { "PrintHighQuality", PR_PRINTFLAGS, 1 },
        { "PrintDescjet", PR_PRINTFLAGS, 2 },
        { "PrintInitPrinter", PR_PRINTFLAGS, 3 },
        { "PrintJobOffset", PR_PRINTFLAGS, 4 },
        { "PrintKyocera", PR_PRINTFLAGS, 5 },
        { "PrintManual", PR_PRINTFLAGS, 6 },

        { "PreviewWhiteOnBlack", PR_PREVIEWFLAGS, 0 },
        { "PreviewRuler", PR_PREVIEWFLAGS, 1 },
        { "PreviewNoStatus", PR_PREVIEWFLAGS, 2 },

        { "TexClear7", PR_TEXFLAGS, 0 },
        { "Tex8BitIn", PR_TEXFLAGS, 1 },
        { "Tex8BitOut", PR_TEXFLAGS, 2 },
        { "TexBatchMode", PR_TEXFLAGS, 3 },
        { "TexNoEMS", PR_TEXFLAGS, 4 },
        { "TexErrorStopMode", PR_TEXFLAGS, 5 },
        { "TexIniTex", PR_TEXFLAGS, 6 },
        { "TexNonstopMode", PR_TEXFLAGS, 7 },
        { "TexControlChars", PR_TEXFLAGS, 8 },
        { "TexScrollMode", PR_TEXFLAGS, 9 },
        { "TexPlain", PR_TEXFLAGS, 10 },
        { "TexGerman", PR_TEXFLAGS, 11 },

        { "" },

    };

    int i;
    char *cp;
    char buffer[EDIT_WIDTH];
    unsigned long flags;

    if (!_check_len(*src,0)) return FALSE;

    for (cp=buffer; **src && **src!=' '; ++*src)
        *cp++ = **src;
    *cp = 0;

    for (i=0; ; ++i) {
        if (strcmp(buffer, FlagArray[i].Name)==0) {
            flags = GetProfileLong( FlagArray[i].Offset );
            if (flags & (1L<<FlagArray[i].BitNum) )
                return TRUE;
            return FALSE;
        }
        if (FlagArray[i].Name[0]==0) {
            DoMessageBox( DMB_OK, HLP_CMDLINEERR,
                  "Warning", "'%s' is not a binary switch", buffer );
            return FALSE;
        }
    }
}


static BOOL _get_resword(char *dst, char **src ) {

    static struct {
        char *Name;
        unsigned int Offset;
        BOOL isString;
    } ResWords[] = {
        { "InitPath", PR_INIPATH, TRUE },
        { "WordWrap", PR_WORDWRAP, FALSE },
        { "TabStops", PR_TABSTOPS, FALSE },
        { "CurrFile", PR_CURRFILE, FALSE },
        { "SearchString", PR_SEARCHSTRING, TRUE },
        { "ReplaceString", PR_REPLACESTRING, TRUE },
        { "FontPath", PR_DRIVEROPTS(0), TRUE },
        { "LibPath", PR_DRIVEROPTS(1), TRUE },
        { "VirtPath", PR_DRIVEROPTS(2), TRUE },
        { "DviPath", PR_DRIVEROPTS(3), TRUE },
        { "GraphPath", PR_DRIVEROPTS(4), TRUE },
        { "DriverAdditional", PR_DRIVEROPTS(5), TRUE },
        { "FontLoadLimit", PR_DRIVERINTS(0), FALSE },
        { "FontScaling", PR_DRIVERINTS(1), FALSE },
        { "DeviceResolutionX", PR_DRIVERINTS(2), FALSE },
        { "DeviceResolutionY", PR_DRIVERINTS(3), FALSE },
        { "FontResolutionX", PR_DRIVERINTS(4), FALSE },
        { "FontResolutionY", PR_DRIVERINTS(5), FALSE },
        { "PreviewAdditional", PR_PREVIEWADD, TRUE },
        { "PreviewCommand", PR_PREVIEWCOMMAND, TRUE },
        { "TexDefaultDrive", PR_TEXDEFDRIVE, TRUE },
        { "PrintStandAlone", PR_PRINTSTANDALONE, TRUE },
        { "TexCommand", PR_TEXCOMMAND, TRUE },
        { "PeriodicSave", PR_PERISAV, FALSE },
        { "LastTexFile", PR_TEXLASTFILE, FALSE },
        { "CurrTexDirectory", PR_TEXCWD, TRUE },
        { "DosCommand", PR_DOSCOMMAND, TRUE },
        { "TempTexFiles", PR_TMPTEXFILES, TRUE },
        { "ThisFile", PR_RESPONSEFILE, TRUE },
        { "PagesHeight", PR_P_HEIGHT, TRUE },
        { "PagesWidth", PR_P_WIDTH, TRUE },
        { "PagesLeftMargin", PR_P_LMARGIN, TRUE },
        { "PagesLeftMarginAdjust", PR_P_LMADJ, TRUE },
        { "PagesTopMargin", PR_P_TMARGIN, TRUE },
        { "PagesTopMarginAdjust", PR_P_TMADJ, TRUE },
        { "PagesTransformation", PR_P_TRANS, TRUE },
        { "PagesCombineFirst", PR_P_FIRST, TRUE },
        { "PagesCombinePages", PR_P_PAGES, TRUE },
        { "PagesCombineIndex", PR_P_INDEX, TRUE },
        { "PagesBegin", PR_P_BEGIN, TRUE },
        { "PagesEnd", PR_P_END, TRUE },
        { "PagesSkip", PR_P_SKIP, TRUE },
        { "PagesNumber", PR_P_NUMBER, TRUE },
        { "PrintAdditional", PR_PRINTADD, TRUE },
        { "PrintOutput", PR_PRINTOUTPUT, TRUE },
        { "PrintRepeat", PR_PRINTREPEAT, TRUE },
        { "PrintPostRepeat", PR_PRINTPOSTREP, TRUE },
        { "PreviewHorizPos", PR_PREVIEWHP, TRUE },
        { "PreviewVertPos", PR_PREVIEWVP, TRUE },
        { "PreviewGrayScalingX", PR_PREVIEWGX, TRUE },
        { "PreviewGrayScalingY", PR_PREVIEWGY, TRUE },
        { "PreviewAdapter", PR_PREVIEWADAPTER, TRUE },
        { "PreviewColors", PR_PREVIEWCOLORS, TRUE },
        { "PreviewUnit", PR_PREVIEWUNIT, TRUE },
        { "PreviewScalingX", PR_PREVIEWSX, TRUE },
        { "PreviewScalingY", PR_PREVIEWSY, TRUE },
        { "PostscriptOutput", PR_PSCRIPTOUTPUT, TRUE },

        { "" }
    };

    int i;
    char *cp;
    char buffer[EDIT_WIDTH];

    if (!_check_len(*src,0)) return FALSE;

    for (cp=buffer; **src && **src!=' '; ++*src)
        *cp++ = **src;
    *cp = 0;

    for (i=0; ; ++i) {
        if (strcmp(buffer, ResWords[i].Name)==0) {
            if (ResWords[i].isString) {
                cp = GetProfileString( ResWords[i].Offset, &profnull );
                if (!_check_len(cp,0)) return FALSE;
                strcpy(dst, cp);
            } else
                sprintf(dst, "%d", GetProfileInt( ResWords[i].Offset ));
            return TRUE;
        }
        if (ResWords[i].Name[0]==0) {
            /* Look if it is a environment variable. */
            if (buffer[0]=='$') {
                if (cp = getenv(buffer+1)) {
                    if (!_check_len(cp,0)) return FALSE;
                    strcpy(dst, cp);
                } else
                    dst[0]=0;
                return TRUE;
            }
            DoMessageBox( DMB_OK, HLP_CMDLINEERR,
                  "Warning", "'%s' is not an editor switch", buffer );
            return FALSE;
        }
    }
}


static BOOL _issection(char *line) {

    return (line[0]=='[' && line[strlen(line)-1]==']');
}

static BOOL _check_len(char *line, int add) {

    if ((strlen(line)+add) < EDIT_WIDTH) return TRUE;

    line[73]=0;
    DoMessageBox( DMB_OK, HLP_CMDLINEERR,
            "Warning", "Line exceeds %d chars\n"
                       "'%s'", EDIT_WIDTH-1, line );
    return FALSE;
}


/* File buffering routines. */

static char *_buf;
static char *_curr;
static char *_end;

static BOOL _read_map_file(int fptr) {

    unsigned long len = Seek(fptr, 0L, SEEK_END);
    Seek(fptr, 0L, SEEK_START);

    if (len>0xffff) return FALSE;
    _buf = AllocHeap((unsigned)len);
    if(_buf==NULL) return FALSE;
    if (_Read(fptr, _buf, (unsigned)len)!=len) {
        Seek(fptr, 0L, SEEK_START);
        return FALSE;
    }

    _curr = _buf;
    _end = _buf+len-1;
    return TRUE;
}

static void _close_map_file(int fptr) {

    if (fptr) Close(fptr);
    else      free(_buf);
}

static long _seek_map_file(int fptr, unsigned long offset, unsigned int flags) {

    if (fptr!=-1) return Seek(fptr, offset, flags);

    switch(flags) {
        case SEEK_START:    _curr = _buf + offset;
                            break;
        case SEEK_CURRENT:  _curr = _curr + offset;
                            break;
        case SEEK_END:      _curr = _end + offset;
                            break;
    }
    return (_curr-_buf);
}


static unsigned int _read_map_line( int fptr, char _far *buf, unsigned int maxlen ) {

    int i=0, ret;

    if (fptr!=-1)
        return _ReadLine(fptr, buf, maxlen);

    /* Fast read line */

    buf[0] = 0;
    for(i=0; i<maxlen; ++i) {
        if (_curr>_end) break;
        if(*_curr == 0x0d) {
            /* carriage return, skip line feed (0x0a) */
            _curr+=2;
            buf[i++]=0;
            break;
        }
        buf[i] = *_curr++;
    }
    buf[i] = 0;
    return i;

}

/* end of file smakec.c */
